; Wideband Controller for LSU4.9 Bosch sensor
; Uses assembly code for PIC16F18877 microcontroller


;	ERRORLEVEL -302
;	ERRORLEVEL -306

#include <pic16f18877.inc>
;#include <xc.inc>

CONFIG FEXTOSC=OFF
CONFIG RSTOSC=HFINT32
CONFIG CLKOUTEN=OFF
CONFIG CSWEN=OFF
CONFIG FCMEN=OFF
CONFIG MCLRE=ON
CONFIG PWRTE=OFF
CONFIG LPBOREN=OFF
CONFIG BOREN=OFF
CONFIG BORV=HI
CONFIG ZCD=OFF
CONFIG PPS1WAY=ON
CONFIG STVREN=OFF
CONFIG DEBUG=OFF
CONFIG WDTE=OFF
CONFIG WRT=OFF
CONFIG SCANE=not_available
CONFIG LVP=OFF
CONFIG CP=OFF
CONFIG CPD=OFF

 
; Define variables at memory locations

; Bank 0 RAM
IMPEDANCE	equ	020h	; impedance on/off flag
HEAT		equ	021h	; heater value
HEAT_COUNT	equ	022h	; heater counter for PWM
RICHLEAN	equ	023h	; rich lean flag for pressure correction
WIDEBAND0	equ	024h	; wideband value ms
WIDEBAND1	equ	025h	; wideband value ls
S_CURVE		equ	026h	; S curve output
CURRENT_IP0	equ	027h	; IP current (PWM output value) ms	
CURRENT_IP1	equ	028h	; IP current (PWM output value) ls
TWO_DIV		equ	029h	; impedance drive/2
REDUC_4		equ	02Ah	; interrupt decrease rate counter
BATT0		equ	02Bh	; battery volts ms byte
BATT1		equ	02Ch	; battery volts ls byte
RAMP		equ	02Dh	; heater ramp				
COUNT1		equ	02Eh	; counter for interrupt counts
HALFSEC		equ	02Fh	; half second counter
AVERAGE     equ 030h    ; average counter for impedance
VH_EFF0		equ	031h	; heater effective voltage ms
VH_EFF1		equ	032h	; heater effective voltage ls
MIN_COUNT	equ	033h	; minute counter
ERROR_FLG	equ	034h	; error flag
FLAG		equ	035h	; flag indicator
SWINGH0		equ	036h	; voltage swing high across VS impedance ms byte
SWINGH1		equ	037h	; voltage swing high across VS impedance ls byte
SWINGL0		equ	038h	; voltage swing low across VS impedance ms byte
SWINGL1		equ	039h	; voltage swing low across VS impedance ls byte	
SPAN0		equ	03Ah	; impedance measuremant voltage span ms byte
SPAN1		equ	03Bh	; impedance measuremant voltage span ls byte	
HEATX		equ	03Ch	; heater value temporary store
WORK0		equ	03Dh	; working value divisor ms
WORK1		equ	03Eh	; working value divisor ls
WORK3		equ	03Fh	; working value multiplier ms
FOUR_SEC	equ	040h	; four seconds counter
CALC0		equ	041h	; calculation temporary value ms byte
RATE		equ	042h	; ramp rate counter
DIV_2		equ	043h	; divide by 2
VOLT0		equ	044h	; battery voltage startup threshold ms byte
VOLT1		equ	045h	; battery voltage threshold ls byte	
STARTFLAG	equ	046h	; after 2-seconds prehat flag for a dimmer startup indicator LED
UPDATE      equ 047h    ; display update rate counter    
SENSE_IP0	equ	048h	; IP sense ms	
SENSE_IP1	equ	049h	; IP sense ls
VS_AVG0		equ	04Ah	; Vs average ms
VS_AVG1		equ	04Bh	; Vs average ls
TEMP_AV0	equ	04Ch	; temporary ms
TEMP_AV1	equ	04Dh	; temporary ls
TEMP_PWM0	equ 04Eh	; temp PWM ms
TEMP_PWM1	equ 04Fh	; temp PWM ls
VS_IP0		equ	050h	; VS/IP	ms
VS_IP1		equ	051h	; VS/IP ls
TWO_COUNT	equ	052h	; A/D AN2 counter
SWING_FLG	equ	053h	; swing flag
MAX0		equ	054h	; calculation for wideband PWM max value ms byte
MAX1		equ	055h	; calculation for wideband PWM max value ls byte
SUB0		equ	056h	; subtract value
SUB1		equ	057h	; subtract value
DIV1		equ	058h	; division value
MULT1		equ	059h	; multiplier value
LAMBDA0     equ 05Ah    ; lambda 1 calibration flag on or off  
OFFSET		equ	05Bh	; pressure sensor offset from VR6/64
PRESSL0		equ	05Ch	; pressure value Low ms
PRESSL1		equ	05Dh	; pressure value low ls
PRESSH0		equ	05Eh	; pressure value High ms
PRESSH1		equ	05Fh	; pressure value high ls
COUNT64     equ 060h    ; counter for divide by 64 
      
; math routines
TEMP160		equ	062h	; DIV1616U temp ms byte
TEMP161		equ	063h	; ls byte	
TEMP		equ	064h	; temp file
REMB0		equ	065h	; remainder ms
REMB1		equ	066h
AARGB0		equ	067h     ; ms of argument A
AARGB1      equ 068h
AARGB2      equ 069h
AARGB3      equ 06Ah	; ls significant byte of argument A
TEMPB1      equ 06Bh	; temporary
TEMPB0      equ 06Ch
BARGB0      equ 06Dh	; 
BARGB1      equ 06Eh	; most significant byte of argument B
LOOPCOUNT   equ 06Fh  	; loop counter

; All Banks RAM (from 70h to 7Fh)

TEMP_AV0X	equ	070h	; temporary ms
TEMP_AV1X	equ	071h	; temporary ls
I_FLG		equ	072h	; current measured flag
ADDN_S0		equ	073h	; averaging addition ms byts
ADDN_S1		equ	074h	; averging addition ls byte
STORE0		equ	075h	; temp store
STORE1		equ	076h	; temp store
STOREH		equ	077h	; ms byte store
STOREL		equ	078h	; ls byte store
TEMPD0      equ 079h    ; temp ls byte for interrupt
TEMPD1		equ	07Ah	; temp ls byte	
STORE3		equ	07Bh	; timer for A/D
DIM			equ	07Ch	; LED dimmer
STORE		equ	07Dh	; storage
AVG0        equ 07Eh    ; averaging for impedance
AVG1        equ 07Fh    ; averaging ls byte       

; Bank 1 RAM       
CNT_16		equ	0A0h	; counter for BCD conversion
; Binary to BCD usage        
BIN_0		equ	0A1h	; binary value ms
BIN_1       equ 0A2h            
BCD_0		equ	0A3h	; BCD MS
BCD_1		equ	0A4h	; display value 
BCD_2       equ 0A5h        
TEMPN		equ 0A6h

; Bank 2 RAM
HOLD_1      equ 120h    ; lambda value store
HOLD_2      equ 121h    ; lambda value store  
     
;define bits
;STATUS flags    
#define C 0
#define Z 2
; FLASH read and write flags
#define NVMREGS 6    
#define RD 0  
#define WR 1
#define FREE 4
#define LWLO 5
#define WREN 2  
; A/D
#define ADON 7
#define ADGO 0        
; Interrupts
#define IOCIE 4
#define TMR0IE 5
#define TMR0IF 5  
#define GIE 7
#define PEIE 6
; timer
#define TRM1IF 0
; serial Tx,Rx
#define TXEN 5 
#define BRG16 3 
#define SYNC 4 
#define BRGH 2
#define SPEN 7        
        
        
               ; NOTE!!!!
;  **** MPLAB X IDE 
; Add the line below in the Production/Set project configuration/Customise/pic-as Global Options/Additional Options:

; -Wl,-pPor_Vec=0h,-pIsr_Vec=4h,-pStorage_Vec=F000h -Wl,-ACODE=0x00-0x9FF, -Wl,-DCODE=2   then call instruction needs PCLATH change beyond 7FF              
; PSECT = program sections  
               
       	
        PSECT   Storage_Vec,global,class=CODE,delta=2
Storage_Vec:; EEPROM
;_PWM
; table for PWM max for duty cycle driven heater
; max varies with battery supply voltage Duty cycle = (VHeff/Vbatt)**2
; returns duty cycle/255
; 7000h  
    DW	163 	; when 15V supply for A-D value/4 = 255 and for 12V limit
	DW	164 	; when 14.94V for A-D value/4 = 254 and for 12V limit
	DW	165 	; when 14.88V for A-D value/4 = 253 and for 12V limit
	DW	167 	; when 14.82V for A-D value/4 = 252 and for 12V limit
	DW	168    	; when 14.76V for A-D value/4 = 251 and for 12V limit
	DW	169		; when 14.71V for A-D value/4 = 250 and for 12V limit
	DW	171		; when 14.65V for A-D value/4 = 249 and for 12V limit
	DW	172		; when 14.59V for A-D value/4 = 248 and for 12V limit
	DW	174		; when 14.53V for A-D value/4 = 247 and for 12V limit
	DW	175		; when 14.47V for A-D value/4 = 246 and for 12V limit
	DW	176		; when 14.41V for A-D value/4 = 245 and for 12V limit
	DW	178		; when 14.35V for A-D value/4 = 244 and for 12V limit
	DW	179		; when 14.29V for A-D value/4 = 243 and for 12V limit
	DW	181		; when 14.24V for A-D value/4 = 242 and for 12V limit
	DW	182		; when 14.18V for A-D value/4 = 241 and for 12V limit
	DW	184		; when 14.12V for A-D value/4 = 240 and for 12V limit
	DW	185		; when 14.06V for A-D value/4 = 239 and for 12V limit
	DW	187		; when 14.00V for A-D value/4 = 238 and for 12V limit
	DW	188		; when 13.94V for A-D value/4 = 237 and for 12V limit
	DW	190		; when 13.88V for A-D value/4 = 236 and for 12V limit
	DW	192		; when 13.82V for A-D value/4 = 235 and for 12V limit
	DW	193		; when 13.76V for A-D value/4 = 234 and for 12V limit
	DW	195		; when 13.71V for A-D value/4 = 233 and for 12V limit
	DW	197		; when 13.65V for A-D value/4 = 232 and for 12V limit
	DW	198		; when 13.59V for A-D value/4 = 231 and for 12V limit
	DW	200		; when 13.53V for A-D value/4 = 230 and for 12V limit
	DW	202		; when 13.47V for A-D value/4 = 229 and for 12V limit
	DW	204		; when 13.41V for A-D value/4 = 228 and for 12V limit
	DW	205		; when 13.35V for A-D value/4 = 227 and for 12V limit
	DW	207		; when 13.29V for A-D value/4 = 226 and for 12V limit
	DW	209		; when 13.24V for A-D value/4 = 225 and for 12V limit
	DW	211		; when 13.18V for A-D value/4 = 224 and for 12V limit
	DW	213		; when 13.12V for A-D value/4 = 223 and for 12V limit
	DW	215		; when 13.06V for A-D value/4 = 222 and for 12V limit
	DW	217		; when 13.00V for A-D value/4 = 221 and for 12V limit
	DW	219		; when 12.94V for A-D value/4 = 220 and for 12V limit
	DW	221		; when 12.88V for A-D value/4 = 219 and for 12V limit
	DW	223		; when 12.82V for A-D value/4 = 218 and for 12V limit
	DW	225		; when 12.76V for A-D value/4 = 217 and for 12V limit
	DW	227		; when 12.71V for A-D value/4 = 216 and for 12V limit
	DW	229		; when 12.65V for A-D value/4 = 215 and for 12V limit
	DW	231		; when 12.59V for A-D value/4 = 214 and for 12V limit
	DW	233		; when 12.53V for A-D value/4 = 213 and for 12V limit
	DW	236		; when 12.47V for A-D value/4 = 212 and for 12V limit
	DW	238		; when 12.41V for A-D value/4 = 211 and for 12V limit
	DW	240		; when 12.35V for A-D value/4 = 210 and for 12V limit
	DW	242		; when 12.29V for A-D value/4 = 209 and for 12V limit
	DW	245		; when 12.24V for A-D value/4 = 208 and for 12V limit
	DW	247		; when 12.18V for A-D value/4 = 207 and for 12V limit
	DW	250		; when 12.12V for A-D value/4 = 206 and for 12V limit
	DW	252		; when 12.06V for A-D value/4 = 205 and for 12V limit
	DW	255		; when 12.00V for A-D value/4 = 204 and for 12V limit
 	
; S_CURVE:
; wideband value/4 if <D22 then 890mV out
; if >D105 then 30mV output
; start at D22
; S-curve output is effectively a divide by 4 value as it uses the 10-bit PWM and ls 8-bits
; so a D255 value in S_CURVE is 5V/4 or 1.25V max
; 7034h	
	DW	182	; 890mV (0.89/5V) x 1024 ;  = 22 	
	DW	182	;   = 23 
	DW	182	;   = 24 
	DW	182	;   = 25 
	DW	181	;   = 26 
	DW	181	;   = 27 
	DW	181	;   = 28 
	DW	180	;   = 29 
	DW	180	;   = 30 
	DW	180	;   = 31 
	DW	179	;   = 32 
	DW	179	; 870mV  ;  = 33
	DW	179	;   = 34 
	DW	179	;   = 35 
	DW	179	;   = 36 
	DW	178	;   = 37 
	DW	178	;   = 38 
	DW	178	;   = 39 
	DW	177	;   = 40 
	DW	177	;   = 41 
	DW	176	;   = 42 
	DW	175	;   = 43 
	DW	174	;   = 44 
	DW	173	;   860mV  ;  = 45  
	DW	172	;   = 46 
	DW	171	;   = 47 
	DW	170	;   = 48 
	DW	169	;   = 49 
	DW	168	;   = 50 
	DW	167	;   = 51 
	DW	166	;   = 52 
	DW	165	;   = 53 
	DW	163	;   = 54 
	DW	161	;   = 55 
	DW	159	;   = 56 
	DW	158	;   = 57 
	DW	156	;   = 58 
	DW	154	;   = 59 
	DW	153	;   = 60 
	DW	151	;   = 61  
	DW	148	;   = 62 
	DW	146	;   = 63 
	DW	141	;  700mV  ; = 64 
	DW	136	;   = 65  
	DW	124	;   = 66 
	DW	114	;   = 67 
	DW	100	;   = 68 
	DW	92	;  450mV ;  = 69 
	DW	63	;   = 70 
	DW	35	;   = 71
 	DW	26	;  130mV ;  = 72 
	DW	20	;   = 73 
	DW	18	;   = 74 
	DW	16	;   = 75 
	DW	14	;   = 76 
	DW	13	;   = 77 
	DW	12	;   = 78 
	DW	12	;   = 79 
	DW	11	;   = 80 
	DW	11	;   = 81 
	DW	10	;   = 82 
	DW	10	;   = 83 
	DW	10	;   = 84 
	DW	9	;   = 85 
	DW	9	;   = 86 
	DW	9	;   = 87 
	DW	8	;   = 88 
	DW	8	;   = 89 
	DW	8	;   = 90 
	DW	8	;   = 91 
	DW	8	;   = 92 
	DW	8	;   = 93 
	DW	8	;   = 94 
	DW	8	;   = 95 
	DW	8	;   = 96 
	DW	7	;   = 97 
	DW	7	;   = 98 
	DW	7	;   = 99 
	DW	7	;   = 100 
	DW	7	;   = 101 
	DW	6	;   = 102 
	DW	6	;   = 103 
	DW	6	;   = 104 
	DW	6	;   = 105 
	DW	6	;   = 106

; table for pressure compensation/correction
; PRESS_LEAN: ; lean lambda >1
; 7089h
; steps 32hPa/step (900hPa range /28)
    DW  100 ; 0
	DW	99	; 1
	DW	98	; 2
	DW	97	; 3
	DW	96	; 4
	DW	95	; 5
	DW	94	; 6
	DW	93	; 7
	DW	92	; 8
	DW	91	; 9
	DW	90	; 10
	DW	89	; 11
	DW	89	; 12
	DW	89	; 13
	DW	89	; 14
	DW	88	; step 15 for 12% reduction 
    DW	88	; 16
	DW	88	; 17
	DW	88	; 18
	DW	87	; 19
	DW	87	; 20
	DW	86	; 21
	DW	86	; 22
	DW	86	; 23	
	DW	85	; 24
	DW	85	; 25
	DW	85	; 26
	DW	85	; 27
    DW  85  ; 28  15% reduction
    
; PRESS_RICH: ; lambds <1
; 70A6h
	DW	100	; 0
	DW	99	; 1
	DW	98	; 2
	DW	97	; 3
	DW	96	; 4
	DW	95	; 5
	DW	95	; 6
	DW	95	; 7
	DW	94	; 8
	DW	94	; 9
	DW	93	; 10
	DW	93	; 11
	DW	92	; 12
	DW	92	; 13
	DW	92	; 14
	DW	92	; 15   8% reduction 
    DW	91	; 16
	DW	91	; 17
	DW	91	; 18
	DW	90	; 19
	DW	90	; 20
	DW	90	; 21
	DW	89	; 22
	DW	89	; 23	
	DW	89	; 24
	DW	89	; 25
	DW	89	; 26
	DW	89	; 27
    DW  89  ; 28  11% reduction
;.................................................... 
 
; define Power on reset 
; Power-On-Reset entry point
        PSECT   Por_Vec,global,class=CODE,delta=2

Por_Vec:
    goto	_START	
    
    
; INTERRUPT    
    
; define interrupt vector start address
; Interrupt vector
        PSECT   Isr_Vec,global,class=CODE,delta=2
     
Isr_Vec:
 
;INTERRUPT: 
	
	BANKSEL PIR4		; bank 0
	bcf		PIR4,0  	; clear timer 1 flag
    BANKSEL TMR1H
	movlw	0FFH
	movwf	TMR1H		; set ms byte of counter
    clrf    TMR1L
	BANKSEL PORTA
    
; heater PWM driver
HEATER:
	movf	HEAT,w		; transfer new pwm value at end of cycle
    BANKSEL PWM6DCH
    movwf   PWM6DCH
    clrf    PWM6DCL
    BANKSEL PORTA

DIV_4_INTERRUPT:

; divide by 4
	decfsz	REDUC_4,f	; when zero continue
	goto	TEST_VALUE
	movlw	4
	movwf	REDUC_4
	goto	DRIVE 		;(PWM drive)
TEST_VALUE: ; split interrupt into 4-independent sections
	movf	REDUC_4,w
	xorlw	2
	btfsc	STATUS,Z
	goto	COUNT_RUN
	movf	REDUC_4,w
	xorlw	1
	btfsc	STATUS,Z
	goto	CK_IMP		; A/D for span and impedance measurement
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
IMPEDANCE_DRIVE:
; impedance drive output set at start of interrupt to ensure a square wave
; not affected by routine length
; check if impedance is on	
	btfss	IMPEDANCE,0	;
	retfie		; 
	incf	TWO_DIV,f	; divide by 2 for ~2kHz (1.953kHz)
	btfss	TWO_DIV,0
	retfie
	btfss	PORTC,0
	goto	SET6
	bcf		LATC,0		; impedance drive 5V square wave
	retfie      ;
SET6:
	bsf		LATC,0		; impedance drive 5V square wave
	retfie
; check if impedance is on
CK_IMP:
	
; check impedance when SWING_FLG,0 is clear
    btfsc   SWING_FLG,0
    goto    ANA6_1
 
    btfsc   PORTC,0     ; if low then read ANA7 for the lower value
    goto    CHECK_2
; measure voltage across impedance of Nernst cell
; for VS detection and impedance measurement
; A/D conversion
; channel ANA7
ANA7_1:
	
	bsf     SWING_FLG,1	; data flag
    movlw	00000111B	; ANA7
	call	ACQUIRE_AD1
	movwf	SWINGL0
	movf	TEMPD0,w
	movwf	SWINGL1		; ls voltage when RC0 is low 
	retfie

; measure voltage across impedance of Nernst cell
; A/D conversion

CHECK_2:

    btfss   PORTC,0     ; if high then read ANA7 for the higher value
  	goto	ANA6_1
ANA7_2:
    bsf     SWING_FLG,2	; data flag
	movlw	00000111B	; ANA7
	call	ACQUIRE_AD1
	movwf	SWINGH0
	movf	TEMPD0,w
	movwf	SWINGH1		; ls voltage when RC0
; if both swingflags (1 and 2 are set then set 0)  
    btfss   SWING_FLG,1
    retfie
    btfss   SWING_FLG,2
    retfie
	bsf		SWING_FLG,0
	retfie
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
    
ANA6_1:
; check Ip sense every ~10ms

	movlw	1
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	ANA6_2
	movlw	81
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	ANA6_2
	movlw	161
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	retfie

; A/D conversion IP sense
; ANA6
ANA6_2:
	movlw	00000110B	; ANA6
	call	ACQUIRE_AD1
	movwf	SENSE_IP0	; ms byte
	movf	TEMPD0,w
	movwf	SENSE_IP1	
	bsf		FLAG,1		; set when end of acquisition set 
	retfie
; ********************

COUNT_RUN:
	decfsz	COUNT1,f	; counter
	goto	MEASURE		; was POWER
	movlw	244		
	movwf	COUNT1		; 32.768ms at 256 counts/ 31.23ms at count of 244
	decf	DIV_2,f
	btfss	DIV_2,0		; 62.4ms period
	goto	MEASURE		; was POWER
	decfsz	RATE,f		; ramp rate counter
	goto	HALF_DWC
	bsf		FLAG,0		; flag for each rate increment
    bsf     FLAG,5
	movlw	3
	movwf	RATE		; reset rate counter
HALF_DWC:
	decfsz	HALFSEC,f
	goto	MEASURE		; was POWER
	bsf		FLAG,2		; uncleared flag indicates when both voltage swings 
						; across Nernst cell have been made
	movf	FOUR_SEC,w	; four seconds counter
	btfss	STATUS,Z	; when zero do not decrease
	decf	FOUR_SEC,f
	movlw	8
	movwf	HALFSEC
	movf	MIN_COUNT,w	; minute counter (120 half seconds)
	btfss	STATUS,Z	; do not decrease if zero
	decf	MIN_COUNT,f	; decrease to zero minute counter

MEASURE:		 
	decf	TWO_COUNT,f	; decrease counter 
	movf	TWO_COUNT,w
	btfsc	STATUS,Z
	goto	MEAS_P
	movf	TWO_COUNT,w	; measure when count is 127
	xorlw	127
	btfss	STATUS,Z
	goto	HEAT_I_AD

MEAS_P: ; measure pressure every ~1.25s; check if STARTFLAG,0 is set
	btfss	STARTFLAG,0
	goto	HEAT_I_AD;POWER
; Plus terminal
	movlw	00011101B	; AND5
	call	ACQUIRE_AD1
	movwf	PRESSH0		; ms
	movf	TEMPD0,w
	movwf	PRESSH1
; offset AND3
    movlw	00011011B	; AND3
	call	ACQUIRE_AD1
	movwf	PRESSL0		; ms
	movf	TEMPD0,w
	movwf	PRESSL1

HEAT_I_AD:

	movf	TWO_COUNT,w	; measure when count is 180
	xorlw	180
	btfss	STATUS,Z
	goto	POWER

; measure heater current	
; A/D conversion
; channel AND6

	movlw	00011110B	; AND6
	call	ACQUIRE_AD1
	movwf	STORE0		; ms byte
	movf	TEMPD0,w
	movwf	STORE1		; ls byte
	bsf		I_FLG,0		; set flag when read

POWER:; LED
; power/heat/error LED

; dim the LED if heater drive is reduced to 0
	btfsc	ERROR_FLG,5
	goto	DIM_LED
; error/data LED flash when error
	btfss	ERROR_FLG,7
	goto	ERR_OFF

; flash LED
FLASH2:
	btfsc	HALFSEC,2	
	goto	CLR2
	bsf		LATA,3	
	retfie
CLR2:
	bcf		LATA,3
	retfie

ERR_OFF:
; flash data when heater ready otherwise stay lit
	btfsc	FLAG,7			; when set, heater ready
	goto	DATA_1
; power/heater LED. Lower brightness until after the 2Veff has finished 
	btfsc	STARTFLAG,0	
	goto	SET_LED
DIM_LED:
; DIM counter
	movf	DIM,w			; dim counter
	btfss	STATUS,Z
	goto	LED_DIM
	movlw	10			; 1/10 duty cycle for LED
	movwf	DIM
	goto	SET_LED

; check new data flash indication required
DATA_1:
  	btfss	ERROR_FLG,6		; new data flag indication 
	goto	LED_OFF
	bcf		ERROR_FLG,6		; data flag clear for next data ready
	btfsc	HALFSEC,0
	goto	LED_OFF
SET_LED:
	bsf		LATA,3			; LED on
	retfie
LED_DIM:
	decf	DIM,f
LED_OFF:	
	bcf		LATA,3			; error LED off
	retfie
; ****************

DRIVE: ; PWM drive for IP, wideband and S curve output 
; check if timer2 close to overrange. T2TMR increments
	movlw	235
    BANKSEL T2TMR
	subwf	T2TMR,w
	btfsc	STATUS,C		; if carry is clear then timer <=235
	retfie					; do not load PWMs if timer2 is close to overflow

; IP_OUT PWM5
; load pwm for 0-5V out
    BANKSEL PORTA
	movf	CURRENT_IP0,w	; ms byte first
	movwf	STOREH          ; store
	movf	CURRENT_IP1,w	; ls byte
	movwf	STOREL			; store

 ; load PWM5
	BANKSEL CCPR1L          ; PWM
	movf    STOREL,w
    movwf   CCPR5L
    movf    STOREH,w
    movwf   CCPR5H
	BANKSEL PORTA			; bank0
	
; load PWM2 for 1.9-5V out 
; check if timer2 close to overrange. T2TMR increments
	movlw	228
    BANKSEL T2TMR
	subwf	T2TMR,w
	btfsc	STATUS,C		; if carry is clear then timer <=228
	retfie
    
    decfsz  UPDATE,f          ; update rate
    retfie                  ; only update every 100ms
    movlw   195
    movwf   UPDATE

; WIDE_OUT
; if JP3 in or out change RD0 and RD1 
    BANKSEL PORTA
    movlw   01111101B   ; JP3 in so RD1 a low output
    btfsc   PORTC,3     ; same bank as TRISx
    movlw   01111110B   ; JP3 out so RD0 a low output
    movwf   TRISD
    
    btfss   PORTC,3     ; JP3 if in (low) RD0 is analog in
    goto    AN_D03
    
; JP3 out (high) RD1 is analog in
    BANKSEL ANSELD
    movlw   01111110B   ; AN6,5,4,3,2,1
    movwf   ANSELD
    goto    TEST_JP2
AN_D03:
    BANKSEL ANSELD 
    movlw   01111101B   ; AN6,5,4,3,2,0
    movwf   ANSELD

TEST_JP2: 
    BANKSEL PORTA
; if JP2 in, output a lambda of 1 (D556)H22C
    btfsc   PORTC,6         ; if low
    goto    USE_WB
; load D556
    movlw   02H
    movwf   STOREH
    movlw   02CH
    movwf   STOREL
    goto    LOAD_WIDE
    
USE_WB:  
   	movf	WIDEBAND0,w		; ms byte first
   	movwf	STOREH			; store
    movf	WIDEBAND1,w		; ls byte
    movwf	STOREL			; store

; load PWM2
LOAD_WIDE:
   	BANKSEL CCPR2L			; PWM
    movf    STOREL,w
    movwf   CCPR2L
    movf    STOREH,w
    movwf   CCPR2H
	BANKSEL PORTA			; bank0

; S_CURVE output PWM1
; If JP1 is inserted show impedance 
    
; place impedance in narrowband PWM
 	btfsc	PORTC,7     ; JP1 in or out
	goto	NARROW_OUT1
; if FLAG,7 is clear clear values
    btfss   FLAG,7
    goto    NC
    
; if AVERAGE is clear start adding
; if AVERAGE is 64, then divide by 64, value is then ready for display 
    movf    AVERAGE,w
    xorlw   64
    btfsc   STATUS,Z
    goto    DIV_IMPEDE
    movf    AVERAGE,w
    btfss   STATUS,Z 
    goto    DO_AVG
; clear AVG0,AVG1 
    clrf    AVG0
    clrf    AVG1
DO_AVG:   
    incf    AVERAGE,f
	movf	SPAN1,w
	addwf	AVG1,f		; ls PWM byte
    btfsc   STATUS,C
    incf    AVG0,f
	movf	SPAN0,w
	addwf   AVG0,f		; ms PWM byte 
    retfie
   
DIV_IMPEDE: ;divide by 64 
    movlw   6
    movwf   COUNT64
CONTX:
    lsrf    AVG0,f       ; shift right ms byte without carry in   
    rrf     AVG1,f       ; shift right ls byte with carry in
    decfsz  COUNT64,f    ; next shift
    goto    CONTX
      
    movf    AVG1,w
    movwf   STOREL
    movf    AVG0,w
    movwf   STOREH
    clrf    AVERAGE
    goto    IMPEDE
    
NC:
    clrf    STOREL
    clrf    STOREH
    goto    IMPEDE
    
; if JP2 in, output a lambda of 1. ie decimal(92)for a reading of 450mV
 NARROW_OUT1:  
    movf	S_CURVE,w 
    btfss   PORTC,6     ; JP2 if low use lambda =1
    movlw   92          ; for 450mV
	movwf	STOREL
    clrf    STOREH
IMPEDE: 
    BANKSEL CCPR1L		; PWM
    movf    STOREL,w
    movwf   CCPR1L
    movf    STOREH,w
    movwf   CCPR1H
	BANKSEL PORTA		; bank0

; end of interrupt  

	retfie					; return from interrupt
    
 ;***************************************************************************************

    PSECT   _START,global,class=CODE,delta=2
_START:

; set oscillator calibration
    BANKSEL OSCFRQ
	movlw	00000110B	; oscillator calibration value 32MHz
	movwf	OSCFRQ   
; open drain
    BANKSEL ODCONB
    bsf     ODCONB,4    ; PORTB,4 open drain for HC-05 enable
 
; TTL in for RX RB2
    BANKSEL INLVLB
    bcf     INLVLB,2
    
; serial data asynchronous 8-bits
    BANKSEL TX1REG
    bsf     RC1STA,SPEN ; bit 7
    bsf     TX1STA,TXEN ; bit 5 enable transmission
    bcf     TX1STA,SYNC ; bit 4
    bsf     TX1STA,BRGH ; bit 2
    bsf     BAUD1CON,BRG16; bit3 set for baud generation via 16 bits
    movlw   3
    movwf   SP1BRGH     ; load 832d 
    movlw   040h
    movwf   SP1BRGL     ; baud rate at 9600 for 32MHz Fosc
       
; map pwm and Rx,Tx. Peripheral Pin Select (PPS)
    BANKSEL RE2PPS
    movlw   0Eh
    movwf   RD7PPS      ; output PWM6 at PORTD7
    movlw   0DH
    movwf   RE2PPS      ; OUTPUT CCP5,PWM5 at PORTE2
    movlw   0CH
    movwf   RB0PPS      ; OUTPUT CCP4,PWM4 at PORTB0
    movlw   0BH
    movwf   RB5PPS      ; OUTPUT CCP3,PWM3 at PORTB5
    movlw   0AH
    movwf   RC1PPS      ; OUTPUT CCP2,PWM2 at PORTC1
    movlw   09H
    movwf   RC2PPS      ; OUTPUT CCP1,PWM1 at PORTC2
    movlw   10H
    movwf   RB2PPS      ; OUTPUT Tx at PORTB2 (requires 5v to 3.3v divider for HC-05
    BANKSEL RXPPS
    movlw   0BH
    movwf   RXPPS       ; INPUT Rx at PORTB3
 	
; set inputs/outputs
	BANKSEL PORTA		; bank0
; set all outputs low
	clrf	LATA
	clrf	LATB
	clrf	LATC
    clrf    LATD
    clrf    LATE
    
; if RB1 low (S1 pressed)keep RB4 low (EN on HC-05 bluetooth)
; otherwise set RB4 high (open drain)
    btfsc   PORTB,1
    bsf     PORTB,4
    
; weak pullups off/on
	BANKSEL WPUA  		; bank	WPU
	bsf		WPUB,1		; PORTB,1 pullup
    movlw   11001000B   
	movwf   WPUC        ; PORTC,7,6,3 pullups
; set I/O
	BANKSEL TRISA		; bank	TRISA/B/C/D/E (also PORTA-E)
	movlw   11010000B	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	00001010B	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	11111000B
	movwf	TRISC
 ; check JP3 in or out
    movlw   01111101B   ; JP3 in so RD1 a low output
    btfsc   PORTC,3     ; same bank as TRISx
    movlw   01111110B   ; JP3 out so RD0 a low output
    movwf   TRISD

    movlw   00001000B
    movwf   TRISE

; analog inputs, A/D
	BANKSEL ANSELA		; ANSELA/B/C/D/E
	movlw	11010000B	; AN7,AN6,AN4 are analog inputs 
	movwf	ANSELA
	movlw	00000000B   ; no analog
	movwf	ANSELB
	movlw	00110000B	; AN5,4
	movwf	ANSELC
  
    BANKSEL PORTA
    btfss   PORTC,3     ; JP3 if in (low) RD0 is analog in
    goto    AN_D0
    
    BANKSEL ANSELD      ; JP3 out (high) RD1 is analog in
    movlw   01111110B   ; AN6,5,4,3,2,1
    movwf   ANSELD
    goto    AN_E
AN_D0:
    BANKSEL ANSELD
    movlw   01111101B   ; AN6,5,4,3,2,0
    movwf   ANSELD
    
AN_E:    
    movlw   00000000B
    movwf   ANSELE

; set up analog to digital     
	BANKSEL ADPCH		; bank1	
	movlw	00000000B	; channel 
	movwf	ADPCH
    movlw   00011111B   ; for 32MHz use Fosc/32 divide for 1us conversion or anywhere in between up to /128
    movwf   ADCLK
    movlw   01010100B   ; 
    movwf   ADCON2      ; low pass filter 
    bsf     ADCON0,2    ; right justified
	bsf		ADCON0,ADON	; A/D on
	
; timer2 set
	BANKSEL T2PR        ; bank	
    bsf     T2CLKCON,0  ; fosc/4
	movlw	0FEH
	movwf	T2PR		; PWM period register
	movlw	10001001B   ; postscale /10
	movwf	T2CON       ; timer2 on

; timer1 set
    BANKSEL T1CON
	movlw	00100011B	; 8MHz counter from 32MHz
	movwf	T1CON
    movlw   00000010B
    movwf   T1CLK       ; Fosc
    
; PWM set
	BANKSEL CCP1CON	
	movlw	10001111B	; set PWM mode
	movwf	CCP1CON		; enable PWM1 operation
	movwf	CCP2CON		; enable PWM2 operation
	movwf	CCP3CON		; enable PWM3 operation
    movwf	CCP4CON		; enable PWM4 operation
	movwf	CCP5CON		; enable PWM5 operation
     
; PWM for -5V generator
    clrf    CCPR3L
	movlw   2           ; 50% duty
	movwf   CCPR3H
    
; PWM for 30V generator
    clrf    CCPR4L
	movlw   2           ; 50% duty
	movwf   CCPR4H
    
; PWM6     
    BANKSEL PWM6CON
    movlw   10000000B
    movwf   PWM6CON
    clrf    PWM6DCH
    clrf    PWM6DCL 
    BANKSEL CCPTMRS1
    bsf     CCPTMRS1,3  ; select timer 6
    bsf     CCPTMRS1,2
    
; timer6 set
	BANKSEL T6PR        ; bank	
    bsf     T6CLKCON,0  ; fosc/4
	movlw	0FEH
	movwf	T6PR		; PWM period register
	movlw	11110001B   ; divide by 128 and /2
	movwf	T6CON       ; timer6 on    
         
; Initialise
    BANKSEL PORTA		; bank0	
	clrf	IMPEDANCE	; impedance measurement off
	movlw	4
	movwf	REDUC_4		; interrupt decrease rate counter
   	clrf	ERROR_FLG	; error flag
	clrf	HEAT		; heater PWM calculated value
    clrf    HEATX
	clrf	SWING_FLG	; amplitude swing flag
    clrf    CALC0       ; compare calculation ms byte
    clrf    AVERAGE     ; impedance averge counter
    
; set PWM 
	clrf	WIDEBAND0	; wideband value ms
	clrf	WIDEBAND1	; wideband value ls
    
; set S-CURVE ouput to 0
	clrf	S_CURVE
; start flag
	clrf	STARTFLAG	; startup flag
	clrf	DIM			; LED dimming counter
	clrf	I_FLG

; timers
	movlw	3
	movwf	RATE		; ramp rate 187.5ms	
	movlw	244		
	movwf	COUNT1		; 62.5ms counter (x 8 for 500ms)
	movlw	60
	movwf	MIN_COUNT	; minute counter to 60 for 1/2 minute (30 seconds) countdown
	movlw	1           ; set at .5 seconds initially
	movwf	FOUR_SEC	; four seconds (8-half seconds)
	movlw	8
	movwf	HALFSEC		; half second
	
; flags
	clrf	FLAG		; indicator flags
	bsf		FLAG,0		; 187.5ms flag
    clrf    LAMBDA0     ; lambda 1 calibration flag set off  
      
; interrupt enable 
    BANKSEL PIR4
	bcf		PIR4,0      ; TMR1IF timer1 flag off
	bsf		PIE4,0      ; TMR1IE timer 1 interrupt enable
	BANKSEL PORTA
	bsf		INTCON,PEIE	; peripheral interrupt
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		IMPEDANCE,0	; impedance on
    
; wait for voltages to settle
CYC_DELAY:
	movf	FOUR_SEC,w
	btfss	STATUS,Z	; when 0 continue on (counter decreased in interrupt)	
	goto	CYC_DELAY
	
	bcf		INTCON,GIE	; stop interrupt for analog reading

; measure VS/IP offset and so set current to pump cell to 0mA	
; A/D conversion
; ANA4; VS/Ip
ANA4_1:
	movlw	00000100B	; ANA4
	call	ACQUIRE_AD 
    movwf	VS_IP0		; VS/IP ms byte
   	movwf	CURRENT_IP0	; IP current ms
    movf	TEMPD1,w 
    movwf	VS_IP1		; VS/IP ls byte
	movwf	CURRENT_IP1	; IP current ls 
    
; measure input voltage startup voltage set by VR13
; A/D conversion
; AND2
	movlw	00011010B	; AND2
	call	ACQUIRE_AD
	movwf	VOLT0		; ms
	movf	TEMPD1,w	; ls
	movwf	VOLT1	
	
	bsf		INTCON,GIE	; interrupt on
      
; measure battery	
; A/D conversion
MEAS_BATT:

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_YZ:
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_YZ
	call	AND4_1      	; measures battery voltage

; check battery voltage. Only start at above setup voltage (VOLT0,VOLT1)
	movf	VOLT0,w			; battery voltage threshold for start ms byte typically at 13V
	subwf	BATT0,w			; ms battery  
	movwf	CALC0
	movf	VOLT1,w			; threshold ls byte 
	subwf	BATT1,w			; ls 
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
    btfsc	CALC0,7			; if set then battery V < Volt0,Volt1
	goto	MEAS_BATT		; measure battery
    
_RUN1:
; reset four second timer
	movlw	4               ; set at 2 seconds for 2Veff for 2-seconds 
	movwf	FOUR_SEC        ; four seconds counter (8-half seconds)
    bsf     ERROR_FLG,5     ; dim LED on
RUNX1:

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_XYZ:
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_XYZ
	call	AND4_1			; measures battery voltage
    
; Temperature of sensor degrees C -40, -10, 20,  25,  50
; VH effective			standard  7.4, 7.8, 8.2, 8.4, 8.5(max)
; initially set at 2VHeff then to 7.4VHeff
; for 7.4V Veff is D504 or H1F8
; other values calculated by multiplying required effective voltage by 68.2 
; and converting to a hexadecimal value (2-byte)
;(actual calculation: /3 for battery voltage divider then /5 for value compared to A/D reference,
; multiply by 1023 for 10-bit range). Result is the same as x 68.2
		
; start preheat at 2V Veff. Duty is [Veff/battery V] squared
; set Veff startup for ramp up 
	movlw	0       	; initial VH effective for heater 140 for 2V effective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	140 		; initial VH effective for heater 
	movwf	VH_EFF1		; heater effective voltage ls
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT
   
; run 2V Veff startup for 2 seconds

	movf	FOUR_SEC,w
	btfss	STATUS,Z	; when 0 continue on (counter decreased in interrupt)	
	goto	RUNX1

; set Veff startup for ramp up 
	movlw	1       	; initial VH effective for heater D504 or H1F8 for 7.2V effective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	0F8H 		; initial VH effective for heater 
	movwf	VH_EFF1		; heater effective voltage ls
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT
    
    movlw	8            ; set at 4 seconds 
	movwf	FOUR_SEC 
; *********************************************************************************************

; program returns here. Normal program run
_RUN:  
    
; if JP2 in show lambda 1 and bypass and shut down sensor
    
    btfsc   LAMBDA0,1   ; if set then bypass 
    goto    SHOW_LAMBDA1
    btfsc   PORTC,6
    goto    HTR
    bsf     LAMBDA0,1   ; lambda of 1 calibration flag on or off. when set is on  
    call    SET_ERR
    goto    SHOW_LAMBDA1
     
HTR:
    btfss	I_FLG,0     ; current flag is set after new current has been measured
	goto	RAMP_UP
	clrf	I_FLG
  
; check heater current for over 4A (0.4V across 0.1 ohm) equivalent to D82 in A/D
; compare with (D82). If over set error flag (ERROR_FLG,7)and stop heater.

	movf	STORE0,w	; high byte if zero check ls byte
	btfss	STATUS,Z
	goto	SET_ERR1	; not zero so set error
; check for over D82
	movf	STORE1,w
	sublw	82
	btfss	STATUS,C	; if > error
	goto	SET_ERR1

; check if <8 for open circuit
; check after first 4-seconds
	movf	FOUR_SEC,w 	; 
	btfss	STATUS,Z	; 
	goto	RAMP_UP
	btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	RAMP_UP		; bypass o/c test after ramp up
	movf	STORE1,w
	sublw	8
	btfsc	STATUS,C
	goto	SET_ERR1
	bcf		ERROR_FLG,7	; no error so clear flag
	goto	RAMP_UP
SET_ERR1:
    call    SET_ERR
    bcf     ERROR_FLG,5 ; dim LED off
    bsf     ERROR_FLG,7 ; flash error LED
SELF:
	goto	SELF		; cycle without heater control
    
; under or over-current so off
SET_ERR:
    clrf	VH_EFF0
	clrf	VH_EFF1		; effective Voltage at 0
	bsf		ERROR_FLG,7	; set error flag
	clrf	IMPEDANCE	; impedance measurement off
	clrf	HEAT		; heater off
; set PWM 
; wideband to lambda 1 (2.71V)
	movlw	2
	movwf	WIDEBAND0	; wideband value ms
	movlw	2CH
	movwf	WIDEBAND1	; wideband value ls

; S curve to 0.45V (D92)
	movlw	92
	movwf	S_CURVE		; S curve output

; set current pump off 
; at the same value as VS/IP offset	

	movf	VS_IP0,w	; VS/IP ms byte
	movwf	CURRENT_IP0	; IP current ms
	movf	VS_IP1,w	; VS/IP ls byte
	movwf	CURRENT_IP1	; IP current ls 
    return

; check for ramp up phase
RAMP_UP:
    btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	HEAT_CON	; control loop of temperature
	bcf		FLAG,1		; clear ready for next ~10ms
; check 187.5ms flag
	btfss	FLAG,0		; if set 187.5ms flag
	goto	DUTY_CALC	
	bcf		FLAG,0

; ramp up at 73.3mV/187.5ms (0.4V per second maximum ramp rate spec.) 390mV/s
; equivalent to digital value of 5
; starts at 7.4V
; this is a Veff ramp up that needs the PWM to be calculated

	movlw	5           ; 73.3mV x 68.2 see above for calculation of required value under Veff section
	addwf	VH_EFF1,f	; heater effective voltage
	btfsc	STATUS,C
	incf	VH_EFF0,f	; increase ms byte when carry
	call	CK_LIMIT	; 
DUTY_CALC:
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT
; ramp up 
   	goto	HEAT_CON 

CK_LIMIT:
; stop increment at Veff = 13V or Veff = 12V. Use Veff = 13V at start and Veff = 12V after 30s.
; if MIN_COUNT is clear 30 seconds have elapsed and use Veff = 12V as the maximum
; 30 seconds allows 2 x 60 x 200h (max allowable hours at 13V is 200h) or 24000 cold starts
; equivalent to more than 240,000kms if average trip is 10kms or more. Note that warm starts would effectively bypass
; the 13V heating
; A Veff of 12V is allowable continuously

; check minute counter
	movf	MIN_COUNT,w	; minute counter
	btfsc	STATUS,Z
	goto	COMP12		; 12V maximum	

; set 13V as maximum
; compare with 13V [D886, H376]
	movlw	03              ; 13V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	76H             ; 13V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 13V
	return
; Veff > 13V so fix at 13V
	movlw	03
	movwf	VH_EFF0			; set at 13V max
	movlw	76H			;
	movwf	VH_EFF1			; fix at 13V max 
    return

COMP12:
; set 12V as maximum after 60-seconds
; compare with 12V [D818, H332]
	movlw	03              ; 12V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	32H             ; 12V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 12V
	return
; Veff > 12V so fix at 12V
	movlw	03
	movwf	VH_EFF0			; set at 12V max
	movlw	32H
	movwf	VH_EFF1			; fix at 12V max 
	return

DUTY:
; calculation of PWM duty cycle during ramp up
; calculate duty cycle based on Vbatt and Veff
; Vbatt is 15V with H3FF measured 
; duty cycle is (VH effective/VBatt)squared. Result is a fraction between 1 and 0 for a heat value of 
; between 255 and 0
; Method: multiply VH effective by 255 (to gain resolution in result) and divide by VBatt. 
; Then square the value and divide by 255 to get a range of 0-255.

; if Vbatt <= Veff then set HEAT at maximum 
 	movf	BATT0,w			; ms battery V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
    movlw   0BAh
    movf	BATT1,w			; ls batt 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	
	btfsc	CALC0,7			; if set then battery V > V eff
	goto	DO_CALC
	movlw	0FFH             ; set heat at max
	return

DO_CALC:; duty cycle calculation for heater

; calculate duty cycle based on Vbatt and Veff
	movf	VH_EFF0,w
	movwf	AARGB0
	movf	VH_EFF1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	0FFH
	movwf	BARGB1
	call	FXM1616U	; multiply Veff x 255
	
 	movf	BATT0,w		; battery volts divisor
	movwf	BARGB0
	movf	BATT1,w		; min of 4 for 16 bit result
	movwf	BARGB1	
	call	FXD3216U	; divide by battery Voltage

; shift
	movf	AARGB2,w	; numerator
	movwf	AARGB0
	movwf	BARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movwf	BARGB1
	call	FXM1616U	; square the value

; divide by 255
	clrf	BARGB0
	movlw	0FFH	
	movwf	BARGB1
	call	FXD3216U
; result in AARGB3
	movf	AARGB3,w
	return
	
HEAT_CON:
    
; two loops check FLAG,7
; 1. during ramp up do not increase duty, but let ramp up process occur
; 2. after ramp up can alter heater drive at will. Uses lookup table to check if 
; the 12V Veff has been reached and alters the PWM value for control rather than Veff 
; set FLAG,7 when Nernst cell is 300 ohms indicating end of heatup phase
; when driving Nernst cell from RC07 with +/-2.5V ac via a 10k ohm impedance, 
; there is a 684.5mV swing from x 4.7 amplifier both high and low for 300 ohm on Nernst cell. 
; Equivalent to a change of 140 decimal from A/D

; check if first swing values have been measured
	btfss	FLAG,2		; cycle ready flag 
	goto	CURRENT_CONTROL

; take SWINGH0,1 from SWINGL0,1 (or vice versa if negative) to get difference in swing across Nernst cell
; (voltages measured in interrupt)

WAIT_SWING1: ; wait for data
 	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING1
	bcf		INTCON,GIE		; stop interrupt
	clrf    SWING_FLG 
    
; subtract Lower swing from Upper swing
	movf	SWINGL0,w		; ms lower 
	subwf	SWINGH0,w		; ms upper  
	movwf	SPAN0
	movf	SWINGL1,w		; ls lower 
	subwf	SWINGH1,w		; ls upper  
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry

	bsf		INTCON,GIE		; re allow interrupt
	movf	SPAN0,w
	btfss	STATUS,Z		; when ms byte of span is not 0, cell is >>300 ohms
	goto	SET_FF			; maximum heating allowed (either ramp up voltage if FLAG,7 not set
							; or at maximum (12V Veff) for FLAG,7 set.
; check if ramp up or control mode
	btfsc	FLAG,7			; when set use control loop for temperature control
	goto	HEAT_LOOP

CK_SPAN: 
; '140' is span for 300 ohms using 10k drive and 4.7 amplifier gain
    movf	SPAN1,w
	sublw	140             ; >300 ohms, set at > 300 ohms so can use heat control loop to maintain temperature
	btfss	STATUS,C		; if minus (c=0) >300 ohms
	goto	SET_FF			; maximum heating allowed. ramp up voltage if FLAG,7 not set
    bsf     FLAG,7			; end of ramp up, start of heat control
    bcf     ERROR_FLG,5     ; dim off
    bcf     ERROR_FLG,7     ; error off
; change pwm rate
    BANKSEL T6PR            ; bank	
	movlw	11000000B       ; divide by 16 for heat control
    movwf	T6CON           ; timer6 on  
    BANKSEL PORTA

HEAT_LOOP:

    movf    SPAN0,w         ; if span excessive
    btfss   STATUS,Z
    goto    SET_FF
    
; alternate between max and min pwm
    movf    SPAN1,w
    sublw   140
    btfsc   STATUS,C
    goto    ALTERNATE
    movlw   255             ; maximum pwm
    movwf   HEATX
  
ALTERNATE:
    movf    15              ; minimum pwm
    movwf   HEATX
    goto    NO_MAX   
 
SET_FF: ;(heat)
	btfss	FLAG,7			; if heatup ramp bypass
	goto	BYPASS_FF
	movlw	0FFH			; set heat at max	
	movwf	HEATX
	goto	CHECK_MAX
BYPASS_FF:
	movf	HEAT,w			; get heat value
	movwf	HEATX			; place in working register

; check 12V Veff limit
; check if maximum duty for Veff of 12V has been reached
CHECK_MAX:
; if in ramp up heat mode bypass max check
	btfss	FLAG,7		
	goto	CURRENT_CONTROL	
	rrf		BATT0,f         ; divide battery volts by 4
	rrf		BATT1,f
	rrf		BATT0,w
	rrf		BATT1,f

	movf	BATT1,w
	sublw	204
	movlw	204         ; load w ready
	btfsc	STATUS,C	; if plus then set at 204 (12V)
	movwf	BATT1
	comf	BATT1,w		; reverse the order
    addlw   0           ; address for PWM table plus W
    movwf   FSR1L       ; EEPROM address
    movlw   70h
    movwf   FSR1H
    moviw   0[FSR1]    
    
    movwf	STORE		; temp value
	subwf	HEATX,w
	movf	STORE,w		; load limit to w ready 
	btfsc	STATUS,C	; if negative value was ok	
	movwf	HEATX		; limit PWM
NO_MAX:
	movf	HEATX,w
	btfss	STATUS,Z	; when zero show error
	goto	BY_ERRA	
; error/data LED flash when error
	bsf		ERROR_FLG,5 ; heater error when heat reduced to 0
	movwf	HEAT
	goto	CURRENT_CONTROL
BY_ERRA:
	bcf		ERROR_FLG,5	; no error
	movwf	HEAT
	
CURRENT_CONTROL: 
    
; keep current at zero until heater is in closed loop maintaining 780 degrees when FLAG,7 is set

	btfsc	FLAG,7	; 
	goto	CURRENT_LOOP

; use measured VS/IP offset to set current to pump cell to 0mA	
; A/D conversion

	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_1:
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_1
	goto	_RUN

CURRENT_LOOP:

; VS detect assumes 2.5V (average) at ANA7 for 450mV Nernst Cell Voltage. 
; Above 2.5 is rich. 
; Below 2.5V is lean

; VS is (SWINGL0,1 + SWINGH0,1)/2
; add

WAIT_SWING2: ; wait for data
	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING2
	bcf		INTCON,GIE		; stop changes in interrupt
	clrf	SWING_FLG
	bcf		FLAG,1			; IP sense flag
	movf	SWINGL0,w
	addwf	SWINGH0,w
	movwf	VS_AVG0			; Ip sense average ms
	movf	SWINGL1,w
	addwf	SWINGH1,w
	bsf		INTCON,GIE		; allow changes in interrupt
	movwf	VS_AVG1			; Ip sense average ls
	btfsc	STATUS,C		; check carry	
	incf	VS_AVG0,f		; increase when ls byte overflows
; divide by 2
	bcf		STATUS,C		; carry clear
	rrf		VS_AVG0,f		; ms byte
	rrf		VS_AVG1,f		; ls byte

; if above 2.5V then mixture is rich so reduce CURRENT_IP0,IP1 in value
; if below 2.5V then mixture is lean so increase CURRENT_IP0,IP1 in value
; allow a hysteresis range above and below 2.5V because of sharp narrow band sensor output

; compare with 2.5V (H1FF)
; subtract VS_AVG0,1 from H1FF

	movf	VS_AVG0,w		; ms byte 
	sublw	1               ; ms 
	movwf	TEMP_AV0
	movf	VS_AVG1,w		; ls 
	sublw	0FFH            ; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
	iorwf	TEMP_AV0,w		; check if both TEMP_AV0 and TEMP_AV1 are at 0
	btfsc	STATUS,Z		; if zero bypass PWM change
	goto	SET_OUTPUT
	btfsc	TEMP_AV0,7		; if set then VS_AVG0,1 > H1FF so rearrange calculation
	goto	REDUCE
	
    movf	TEMP_AV0,w
	movwf	TEMP_AV0X
	movf	TEMP_AV1,w
	movwf	TEMP_AV1X
    
; decrease hysteresis
; value of 1 = (4.8mV) from VS_AVG0,1 equivalent to 1.02mV on VS
	movlw	5			;  
	subwf	TEMP_AV1,w		; to 'w' for comparison with 0 (ls byte)
	movwf	TEMP_AV1		; equivalent to subwf TEMP_AV1,f but with value in w as well as f 
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry

	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	MULT_INC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	PROP_INC
    
MULT_INC:
; multiply by 2
	bcf		STATUS,C
	rlf		TEMP_AV1X,f
	rlf		TEMP_AV0X,f
	movf	TEMP_AV0X,w
	sublw	3			; if >3 set at 3
	btfsc	STATUS,C
	goto	PROP_INC
SET_MAX_INC:
	movlw	3
	movwf	TEMP_AV0X
	movlw	0FFh
	movwf	TEMP_AV1X

; setup a proportionate increase rate
PROP_INC: 
 ; INCREASE
; add value to PWM
	movf	CURRENT_IP0,w	; ms byte 
	addwf	TEMP_AV0X,w
	movwf	TEMP_PWM0

	movf	TEMP_AV1X,w		; ls byte
	addwf	CURRENT_IP1,w
	movwf	TEMP_PWM1	
	btfsc	STATUS,C		; if carry increase ms byte if 2 or less
	incf	TEMP_PWM0,f

	movf	TEMP_PWM0,w
	sublw	3               ; if >3 do not increase
	btfsc	STATUS,C
	goto	TRANS_PWM		; place into current drive PWM

SET_MAX:
; set at 3FF
	movlw	0FFh
    movwf   CURRENT_IP1
  	movwf	WIDEBAND1
    movwf   TEMP_PWM1
	movlw	3
    movwf   CURRENT_IP0
  	movwf	WIDEBAND0
    movf    TEMP_PWM0
	bsf		ERROR_FLG,7		; error LED on at limit
	goto	SET_OUTPUT

REDUCE:
; subtract H1FF (2.5V) from VS_AVG0,1
	movlw	1
	subwf	VS_AVG0,w		; ms byte 
	movwf	TEMP_AV0
	movlw	0FFH 			; ls 
	subwf	VS_AVG1,w		; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry

    movf	TEMP_AV0,w
	movwf	TEMP_AV0X
	movf	TEMP_AV1,w
	movwf	TEMP_AV1X
    
; decrease hysteresis
; Value of 1 = (4.8mV) from VS_AVG0,1 equivalent to 1.02mV on Vs
	movlw	5			;  
	subwf	TEMP_AV1,w		; w for comparison (ls byte)
	movwf	TEMP_AV1		;  
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
	
	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	MULT_DEC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	PROP_DEC

MULT_DEC:
; multiply by 2
	bcf		STATUS,C
	rlf		TEMP_AV1X,f
	rlf		TEMP_AV0X,f
	movf	TEMP_AV0X,w
	sublw	3			; if >3 set at 3
	btfsc	STATUS,C
	goto	PROP_DEC
SET_MAX_DEC:
	movlw	3
	movwf	TEMP_AV0X
	movlw	0FFh
	movwf	TEMP_AV1X
    
; setup a proportionate decrease rate
PROP_DEC:
; REDUCE VALUE
	movf	TEMP_AV0X,w		; ms byte 
	subwf	CURRENT_IP0,w
	movwf	TEMP_PWM0

	movf	TEMP_AV1X,w
	subwf	CURRENT_IP1,w	; ls 
	movwf	TEMP_PWM1
	btfss	STATUS,C
	decf	TEMP_PWM0,f		; decrease ms if carry
	btfsc	TEMP_PWM0,7		; if set then negative so set at 0
	goto	ZERO_SET
; transfer to PWM
TRANS_PWM:
	bcf		INTCON,GIE	
	movf	TEMP_PWM0,w		; ms byte 
	movwf	CURRENT_IP0
	movf	TEMP_PWM1,w		; ls byte 
	movwf	CURRENT_IP1
	goto	SET_OUTPUT

ZERO_SET:
    bcf     INTCON,GIE
	clrf    WIDEBAND1
    clrf    WIDEBAND0
    clrf    TEMP_PWM0
    clrf    TEMP_PWM1
    goto    TRANS_PWM
    
; allow 10ms for pump action
    
   
; Nernst cell at 450mV is set to 2.5V

SET_OUTPUT: 
   bcf		ERROR_FLG,7		; ERROR LED off
   goto     CK_CURRENT 
 	
; When ANA7 (Vs - 0.45V x 4.7V) is close to 2.5V (0.45V across Nernst cell)
; VS_AVG0,1 is the average AN0 value
; TEMP_AV1 is the offset from 2.5V 
; so as TEMP_AV1 is zero load values

CK_CURRENT: 
; when data is correct, flash DATA LED 
    bsf		INTCON,GIE		; resume interrupt
	bsf		ERROR_FLG,6		; data flag
 
READINGS:
       
; get (amplified) IP sense value and subtract from VS_IP
; SENSE_IP0,1 is IP sense at ANA6
; subtract VS_IP from IP Sense
    bcf     STATUS,GIE      ; interrupt off
	movf	VS_IP0,w		; ms VS_IP 
   	subwf	SENSE_IP0,w		; ms IP sense  
	movwf	TEMPB0
	movf	VS_IP1,w		; ls  
    subwf	SENSE_IP1,w		; ls  
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry
	btfss	TEMPB0,7		; if set then VS_IP > IP sense so rearrange calculation
	goto	SET_WIDE_LEAN 

; subtract 
	movf	SENSE_IP0,w		; ms IP sense 
	subwf	VS_IP0,w		; ms VS_IP 
	movwf	TEMPB0
	movf	SENSE_IP1,w		; ls  
	subwf	VS_IP1,w		; ls 
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry	
    
SET_WIDE_RICH: ;(Lambda <1.016)
    bsf     STATUS,GIE      ; interrupt on
; pressure correction
	bsf		RICHLEAN,0		; sets for rich correction
	call	PRESSURE 
    bsf     STATUS,GIE      ; interrupt on
     
; values obtained from calculated table. See wideband table at end of file
; <= D161?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	161			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 > 151 
	goto	RICH1
; <= 245?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	245			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 245
	goto	RICH2
; <= D358?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	66H			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > D348
	goto	RICH3
; <= D587?
	movf	TEMPB0,w 		; ms 
	sublw	2
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	4BH			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 596
	goto	RICH4
; above D587 is out of range so error
    
    movlw   2
    movwf   TEMPB0
    movlw   4BH
    movwf   TEMPB1
    bsf     ERROR_FLG,7     ; error flash
    goto    RICH4

RICH1:
; 1.016-0.9
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)

				
; TEMPB0,1 between D0 and D161, Lambda is between 1.016 and 0.9, output for wideband is from D283 to D180 
; 1.39V to 0.88V.
; Calculation for wideband PWM output is D277 - (((TEMPB0,1- 0')/D161) x D97). Note D97 = D277-D180

	movlw	01			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	1BH
	movwf	MAX1			; max value D269 = H010D
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	161			; divide value
	movwf	DIV1
	movlw	97

	goto	CALC_PWM_WIDE_R

RICH2:
; 0.9-0.85
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)
;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
				
; TEMPB0,1 between D161 and D245, Lambda is between 0.9 and 0.85, output for wideband is from D180 to D135 
; 0.88V to 0.66V.
; Calculation for wideband PWM output is D180 - (((TEMPB0,1-D161)/D84) x D45).
; Note D45 = D180-D135; D94 is D245-D161

	movlw	00			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	180
	movwf	MAX1			; max value 
	movlw	161			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	84			; divide value
	movwf	DIV1
	movlw	45

	goto	CALC_PWM_WIDE_R

RICH3:
; 0.85-0.8

;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)

				
; TEMPB0,1 between D245 and D358, Lambda is between 0.85 and 0.8, output for wideband is from D135 to D90 
; 0.66V to 0.44V.
; Calculation for wideband PWM output is D135 - (((TEMPB0,1-D245)/D113) x D45).
; Note D45 = D135-D90; D113 is D358-D245

	movlw	00			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	135
	movwf	MAX1			; max value 
	movlw	245			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	113			; divide value
	movwf	DIV1
	movlw	45

	goto	CALC_PWM_WIDE_R

RICH4:
; 0.8-0.7

;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)
;*0.7		-1.82 mA	2.87V (D587)	0V (D0)			
				
; TEMPB0,1 between D358 and D596, Lambda is between 0.8 and 0.7, output for wideband is from D90 to D0 
; 0.44V to 0.0V.
; Calculation for wideband PWM output is D90 - (((TEMPB0,1-D358)/D238) x D90).
; Note D90 = D90-D0; D248 is D596-D358

	movlw	0			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	90
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	66H
	movwf	SUB1			; subtract value ls
	movlw	238			; divide value
	movwf	DIV1
	movlw	90

	goto	CALC_PWM_WIDE_R

SET_WIDE_LEAN:
    bsf     STATUS,GIE      ; interrupt on
; pressure correction
	bcf		RICHLEAN,0		; sets for lean correction
	call	PRESSURE
    bsf     STATUS,GIE      ; interrupt on
      
; <= D18?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	18			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 18 
	goto	LEAN1
; <= D37?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	37			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN2
; <= D55?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	55			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN3
; <= D73?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	73			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN4
; <= D91?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	91			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN5
; <= D109?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	109			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN6
; <= D128?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	128			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN7
; <= D146?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	146			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN8
; <= D164?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	164			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN9
; <= D183?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	183			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN10
; <= D201?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	201			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN11
; <= D219?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	219			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN12
; <= D238?
	movf	TEMPB0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	238			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN13
; <= D257?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	1			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN14
; <= D276?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	14			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN15
; <= D295?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	27			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN16
; <= D306?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	32			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN17
; <= D314?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	3AH			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN18
; <= D334?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	4EH			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN19
; <= D345?
	movf	TEMPB0,w 		; ms 
	sublw	1
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	59H			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN20
; above D345 is out of range so set at 5V
    
    movlw   1
    movwf   TEMPB0
    movlw   59H
    movwf   TEMPB1
    bsf     ERROR_FLG,7     ; error out of range
    goto    LEAN20
 
LEAN1:
; 1.016 to 1.026
; TEMPB0,1 between D0 and D18, Lambda is between 1 and 1.026, output for wideband is from D277 to D292 
; 1.43V to 1.39V.
; Calculation for wideband PWM output is D292 - (((D18-TEMPB0,1)/D18) x D15).
; Note D15 = D292-D277; D18 is D18-D0

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	24H
	movwf	MAX1			; max value D285
	movlw	18			; subtract value from TEMPB0,1
	movwf	SUB1			; subtract value ls	
	clrf	SUB0			; subtract value ms
	movlw	18			; divide value
	movwf	DIV1
	movlw	15

	goto	CALC_PWM_WIDE
LEAN2:
; 1.026 to 1.0534
; TEMPB0,1 between D18 and D37, output for wideband is from D292 to D317 
; Calculation for wideband PWM output is D317 - (((D37-TEMPB0,1)/D19) x D25).
; Note D25 = D317-D292; D19 is D37-D18

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	3DH
	movwf	MAX1			; max value D317
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	37
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	25

	goto	CALC_PWM_WIDE
LEAN3:
; 1.0534 to 1.0823
; TEMPB0,1 between D37 and D55, output for wideband is from  D317 to D343
; Calculation for wideband PWM output is D343 - (((D55-TEMPB0,1)/D18) x D26).
; Note D26 = D343-D317; D18 is D55-D37

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	57H
	movwf	MAX1			; max value D343
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	55
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	26

	goto	CALC_PWM_WIDE
LEAN4:
; 1.0823 to 1.112
; TEMPB0,1 between D55 and D73, output for wideband is from D343 to D370
; Calculation for wideband PWM output is D370 - (((D73-TEMPB0,1)/D18) x D27).
; Note MULT D27 = D370-D343; DIV D18 is D73-D55

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	72H
	movwf	MAX1			; max value D370
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	73
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	27

	goto	CALC_PWM_WIDE
LEAN5:
; 1.112 to 1.1445
; TEMPB0,1 between D73 and D91, output for wideband is from D399 to D370
; Calculation for wideband PWM output is D399 - (((D91-TEMPB0,1)/D18) x D29).
; Note MULT D29 = D399-D370; DIV D18 is D91-D73

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	8FH
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	91
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	29

	goto	CALC_PWM_WIDE
LEAN6: 
; 1.1445 to 1.18
; TEMPB0,1 between D91 and D109, output for wideband is from D399 to D431
; Calculation for wideband PWM output is D431 - (((D109-TEMPB0,1)/D18) x D32).
; Note MULT D32 = D431-D399; DIV D18 is D109-D91

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0AFH
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	109
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	32

	goto	CALC_PWM_WIDE
LEAN7: 
; 1.18 to 1.214
; TEMPB0,1 between D109 and D128, output for wideband is from D431 to D462
; Calculation for wideband PWM output is D462 - (((D128-TEMPB0,1)/D19) x D31).
; Note MULT D31 = D462-D431; DIV D19 is D128-D109

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0CEH
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	128
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	31

	goto	CALC_PWM_WIDE
LEAN8: 
; 1.214 to 1.2516
; TEMPB0,1 between D128 and D146, output for wideband is from D462 to D495
; Calculation for wideband PWM output is D495 - (((D146-TEMPB0,1)/D18) x D33).
; Note MULT D33 = D495-D462; DIV D18 is D146-D128

	movlw	1			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0EFH
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	146
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	33

	goto	CALC_PWM_WIDE
LEAN9: 
; 1.2516 to 1.2916
; TEMPB0,1 between D146 and D164, output for wideband is from D531 to D495
; Calculation for wideband PWM output is D531 - (((D164-TEMPB0,1)/D18) x D36).
; Note MULT D36 = D531-D495; DIV D18 is D164-D146

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	13H
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	164
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	36

	goto	CALC_PWM_WIDE
LEAN10: 
; 1.2916 to 1.33
; TEMPB0,1 between D164 and D183, output for wideband is from D531 to D566
; Calculation for wideband PWM output is D566 - (((D183-TEMPB0,1)/D19) x D35).
; Note MULT D35 = D566-D531; DIV D19 is D183-D164

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	36H
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	183
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	35

	goto	CALC_PWM_WIDE
LEAN11: 
; 1.33 to 1.38
; TEMPB0,1 between D183 and D201, output for wideband is from D611 to D566
; Calculation for wideband PWM output is D611 - (((D201-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D611-D566; DIV D18 is D201-D183

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	63H
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	201
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	45

	goto	CALC_PWM_WIDE
LEAN12: 
; 1.38 to 1.43
; TEMPB0,1 between D201 and D219, output for wideband is from D611 to D656
; Calculation for wideband PWM output is D656 - (((D219-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D656-D611; DIV D18 is D219-D201

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	90
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	219
	movwf	SUB1			; subtract value ls
	movlw	18			; divide value
	movwf	DIV1
	movlw	45

	goto	CALC_PWM_WIDE
LEAN13: 
; 1.43 to 1.48
; TEMPB0,1 between D219 and D238, output for wideband is from D701 to D656
; Calculation for wideband PWM output is D701 - (((D238-TEMPB0,1)/D19) x D45).
; Note MULT D45 = D701-D656; DIV D19 is D238-D219

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0BDH
	movwf	MAX1			; max value 
	movlw	0			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	238
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	45

	goto	CALC_PWM_WIDE
LEAN14: 
; 1.48 to 1.5346
; TEMPB0,1 between D238 and D257, output for wideband is from D701 to D750
; Calculation for wideband PWM output is D750 - (((D257-TEMPB0,1)/D19) x D49).
; Note MULT D49 = D750-D701; DIV D19 is D257-D238

	movlw	2			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0EEH
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	1
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	49

	goto	CALC_PWM_WIDE
LEAN15: 
; 1.5346 to 1.594
; TEMPB0,1 between D257 and D276, output for wideband is from D803 to D750
; Calculation for wideband PWM output is D803 - (((D276-TEMPB0,1)/D19) x D53).
; Note MULT D53 = D803-D750; DIV D19 is D276-D257

	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	23
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	14H
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	53

	goto	CALC_PWM_WIDE
LEAN16: 
; 1.594 to 1.658
; TEMPB0,1 between D276 and D295, output for wideband is from D803 to D861
; Calculation for wideband PWM output is D861 - (((D295-TEMPB0,1)/D19) x D53).
; Note MULT D58 = D861-D803; DIV D19 is D295-D276

	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	5DH
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	27
	movwf	SUB1			; subtract value ls
	movlw	19			; divide value
	movwf	DIV1
	movlw	58

	goto	CALC_PWM_WIDE
LEAN17: 
; 1.658 to 1.7
; TEMPB0,1 between D295 and D306, output for wideband is from D898 to D861
; Calculation for wideband PWM output is D898 - (((D306-TEMPB0,1)/D11) x D31).
; Note MULT D37 = D898-D861; DIV D11 is D306-D295

	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	82H
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	32H
	movwf	SUB1			; subtract value ls
	movlw	11			; divide value
	movwf	DIV1
	movlw	37

	goto	CALC_PWM_WIDE
LEAN18: 
; 1.7 to 1.73
; TEMPB0,1 between D306 and D314, output for wideband is from D898 to D925
; Calculation for wideband PWM output is D925 - (((D314-TEMPB0,1)/D8) x D27).
; Note MULT D27 = D925-D898; DIV D8 is D314-D306
	
	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	9DH
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	3AH
	movwf	SUB1			; subtract value ls
	movlw	8			; divide value
	movwf	DIV1
	movlw	27

	goto	CALC_PWM_WIDE
LEAN19: 
; 1.73 to 1.80
; TEMPB0,1 between D314 and D334, output for wideband is from D925 to D988
; Calculation for wideband PWM output is D988 - (((D334-TEMPB0,1)/D20) x D63).
; Note MULT D63 = D988-D925; DIV D20 is D334-D314

	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0DCH
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	4E
	movwf	SUB1			; subtract value ls
	movlw	20			; divide value
	movwf	DIV1
	movlw	63

	goto	CALC_PWM_WIDE
LEAN20: 
; 1.80 to 1.84
; TEMPB0,1 between D334 and D345, output for wideband is from D1023 to D988
; Calculation for wideband PWM output is D1023 - (((D345-TEMPB0,1)/D11) x D35).
; Note MULT D35 = D1023-D988; DIV D11 is D345-D334

	movlw	3			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	0FFH
	movwf	MAX1			; max value 
	movlw	1			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	059H
	movwf	SUB1			; subtract value ls
	movlw	11			; divide value
	movwf	DIV1
	movlw	35

	goto	CALC_PWM_WIDE

CALC_PWM_WIDE_R:
; RICH Calculation
	movwf	MULT1
	bcf		ERROR_FLG,7		; ERROR LED off	
; take subtract value from TEMPB0,1
	movf	SUB0,w 			; ms 
	subwf	TEMPB0,w		; subtract value
	movwf	AARGB0
	movf	SUB1,w			; ls  
	subwf	TEMPB1,w		; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	
	goto	MULT_ALL

CALC_PWM_WIDE:
; lean calculation
	movwf	MULT1
	bcf		ERROR_FLG,7		; ERROR LED off	
; take  TEMPB0,1 from subtract value 
	movf	TEMPB0,w 		; ms 
	subwf	SUB0,w			; subtract value
	movwf	AARGB0
	movf	TEMPB1,w		; ls  
	subwf	SUB1,w			; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	

MULT_ALL:
; multiply
	clrf	BARGB0
	movf	MULT1,w			; multiply value
	movwf	BARGB1
	call	FXM1616U		; multiply

; divide
	movf	DIV1,w
	movwf	BARGB1
	clrf	BARGB0
	call	FXD3216U		; divide

; subtract from maximum 
	
	movf	AARGB2,w 		; ms 
	subwf	MAX0,f			; subtract value
	movf	AARGB3,w		; ls  
	subwf	MAX1,f			; subtract value ls  
	btfss	STATUS,C
	decf	MAX0,f	
    
; use values as unchanged to produce narrow band out, so transfer values to other ADDN_S0,1 for narrow out calculation
  
SET1:  
     
    movf    MAX0,w
    movwf   ADDN_S0
  
    movf    MAX1,w
    movwf   ADDN_S1
      
; recalculate for output instead of going from 0V to 5V for 0.7 to 1.84 lambda
; to 1.9V to 5V for 0.7 to 1.84 lambda 
; multiply by 0.62 (span of 3.1V (1.9V to 5V) with 5V max)and add 1.9V (388D, 184H)    
; 556d is lambda of 1    
; multiply by 62
    movf    MAX0,w
    movwf   AARGB0
    movf    MAX1,w
    movwf   AARGB1
	clrf	BARGB0
	movlw	62      		; multiply value
	movwf	BARGB1
	call	FXM1616U		; multiply

; divide by 100 (overall this equals x 0.62)
	movlw	100
	movwf	BARGB1
	clrf	BARGB0
	call	FXD3216U		; divide

; add 1.9V (184H)
	movlw   1
	addwf	AARGB2,f 		; ms 
	movlw   84H
	addwf	AARGB3,f		; ls  
	
	btfsc	STATUS,C
	incf	AARGB2,f
    
    movf    AARGB2,w
    movwf   MAX0
    movf    AARGB3,w
    movwf   MAX1
  
; if JP2 in for lambda of 1 calibrate ( for ADDN_S0 and ADDN_S1 use 269 for digital value (0.7 lambda is 0V 1.84 lambda 5V)
; so lambda 1 is 1.315V or 269d (10Dh) digital for 10-bits.
SHOW_LAMBDA1:    ;  
    btfsc   PORTC,6
    goto    BLU
; 556d is lambda 1 for bluetooth
    movlw   02h
    movwf   MAX0
    movlw   2Ch
    movwf   MAX1
    
; load 269d, for narrowband
    movlw   01h
    movwf   ADDN_S0
    movlw   0Dh
    movwf   ADDN_S1
    
; ...................................................................................
; Bluetooth values
    
; For Lambda value    
; use MAX0,MAX1 (388D to 1023D range) values for bluetooth  
; divide by 5.55 (x 100 and divide by 555) for 70D to 184D (decimal point adjust decimal point in display for 0.7 to 1.84 lambda)
; then convert to BCD and send via serial and ASCII or utf8 (3xh) where 3 is prefix ms nibble and 'x' is 0-9) as one bcd byte per value.
; 
; For Air/Fuel ratio 
; use AF1 setting at TP7/VR7 if JP3 is in and AF2 at TP8/VR8 if JP3 is out
; TP7 and TP8 are set so for example for 14.7AF ratio set at 1.47V, for 15.5 A/F ratio set TP8 at 1.55V   
; using MAX0,MAX1 multiply by A/D value (eg. 14.7 A/F is 301D for 1.47V at TP7) and divide by 204. Then x100/555  
; then convert to BCD and send via serial and utf8 (3xh) where x is 0-9) as one bcd byte per value.
;
;      
BLU: 
; do lambda calculations
    movf    MAX0,w
    movwf   AARGB0
    movf    MAX1,w
    movwf   AARGB1
    clrf    BARGB0
    movlw   100
    movwf   BARGB1
    call    FXM1616U
    
; divide by 555 Decimal
    movlw   2
    movwf   BARGB0
	movlw   2Bh
	movwf	BARGB1
	call	FXD3216U		; divide
; AARGB2, AARGB3 has 16-bit result
    
    movf    AARGB2,w
    BANKSEL BIN_0
    movwf   BIN_0
    BANKSEL AARGB3
    movf    AARGB3,w
    BANKSEL BIN_1
    movwf   BIN_1
    call    BCD             ; convert to packed BCD
; result in BCD_0, BCD_1 and BCD_2
; place values in hold for later 
    BANKSEL BCD_1
    movf    BCD_1,w
    BANKSEL TX1STA  ; using bank 2 for HOLD values
    movwf   HOLD_1
    BANKSEL BCD_1
    movf    BCD_2,w
    BANKSEL TX1STA  ; using bank 2 for HOLD values
    movwf   HOLD_2
     
;  do A/F calculations    
; A/F
    BANKSEL PORTC
    movlw   00010100B       ; initally ANC4 for AF2
    btfss   PORTC,3         ; if high AF2, low AF1
    movlw   00010101B       ; ANC5 for AF1
    bcf     INTCON,GIE
    call    ACQUIRE_AD      ; get value at AF settings
     
; w has LSB and TEMPD1 has MSB of A/D conversion
; multiply MAX0,MAX1 by A/D at ANC4 or ANC5
; multiply
    movwf   BARGB0          ; ms byte of A/D
    bsf     INTCON,GIE
    
    movf    TEMPD1,w
    movwf   BARGB1          ; ls byte of A/D
    movf    MAX0,w
    movwf   AARGB0
    movf    MAX1,w
    movwf   AARGB1
 	call	FXM1616U		; multiply

; divide by 204 Decimal
    movlw   0
    movwf   BARGB0
	movlw   204
	movwf	BARGB1
	call	FXD3216U		; divide
; AARGB2 and AARGB3 has 16-bit result
    ; x 100
    movf    AARGB2,w
    movwf   AARGB0
    movf    AARGB3,w
    movwf   AARGB1
    clrf    BARGB0
    movlw   100
    movwf   BARGB1
    call    FXM1616U
    
; divide by 555 Decimal
    movlw   2
    movwf   BARGB0
	movlw   2Bh
	movwf	BARGB1
	call	FXD3216U		; divide
; AARGB2, AARGB3 has 16-bit result
    
    movf    AARGB2,w
    BANKSEL BIN_0
    movwf   BIN_0
    BANKSEL AARGB3
    movf    AARGB3,w
    BANKSEL BIN_1
    movwf   BIN_1
    call    BCD             ; convert to packed BCD
    
; result in BCD_0, BCD_1 and BCD_2
   ; send via Bluetooth 
   ; TX1STA,1 set when buffer empty
   ; TX1REG is data register
   
; Air/Fuel data sent first
    BANKSEL TX1STA
WAIT3: ; load ms bcd byte   
    btfss   TX1STA,1    ; if buffer empty, load value
    goto    WAIT3
    BANKSEL BCD_1
    movf    BCD_1,w     ; process ms byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; prefix 3 for utf8/ASCII
    BANKSEL TX1REG
    movwf   TX1REG    
 
WAIT2: ; load next mid bcd byte
    btfss   TX1STA,1
	goto	WAIT2	
    BANKSEL BCD_2
    swapf   BCD_2,w     ; swap nibbles over to get next packed BCD byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; prefix 3 for utf8/ASCII
    BANKSEL TX1REG
    movwf   TX1REG

WAIT1: ; load ls bcd byte   
    btfss   TX1STA,1    ; if buffer empty, load value
    goto    WAIT1
    BANKSEL BCD_2
    movf    BCD_2,w     ; process ls byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; add prefix 3 for utf8/ASCII
    BANKSEL TX1REG
    movwf   TX1REG   
  
; Lambda value next
       
WAIT6: ; load ms bcd byte   
    btfss   TX1STA,1    ; if buffer empty, load value
    goto    WAIT6
    movf    HOLD_1,w    ; process ms byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; prefix 3 for utf8/ASCII
    movwf   TX1REG    
 
WAIT5: ; load next mid bcd byte
    btfss   TX1STA,1
	goto	WAIT5	
    swapf   HOLD_2,w    ; swap nibbles over to get next packed BCD byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; prefix 3 for utf8/ASCII
    movwf   TX1REG

WAIT4: ; load ls bcd byte   
    btfss   TX1STA,1    ; if buffer empty, load value
    goto    WAIT4
    movf    HOLD_2,w    ; process ls byte
    andlw   00001111B   ; remove ms nibble
    iorlw   00110000B   ; add prefix 3 for utf8/ASCII
    movwf   TX1REG   

LINE1: ; send end of line signal (end of data stream)
    btfss   TX1STA,1
    goto    LINE1
    movlw   0Ah         ; line feed  ( \n )
    movwf   TX1REG
    
; end of bluetooth serial data loading
; ....................................................................................    

; place wideband into PWM
PLACE_WB:
    BANKSEL PORTA
	bcf		INTCON,GIE		; stop changes in interrupt
 	movf	MAX0,w
    movwf	WIDEBAND0		; ms PWM byte for wideband output	
	movf	MAX1,w
    movwf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		INTCON,GIE		; allow changes in interrupt

NARROW_BAND:
    
; S_CURVE output
; Calculation for narrowband S-CURVE
; divide WIDEBAND0,1(ADDN_SO, ADDN_S1) by 4 
WIDE_4:
	bcf		STATUS,C
	rrf		ADDN_S0,f		; divide by 2
	rrf		ADDN_S1,f
	bcf		STATUS,C
	rrf		ADDN_S0,f		; divide by 2 again		
	rrf		ADDN_S1,f		; 

; check if range is < 22 or > if 105 
; less than 22
	movlw	22
	subwf	ADDN_S1,f
	btfss	STATUS,C		; if minus then less than 22
	clrf	ADDN_S1			; set at 0 if less
; compare to 83 (105 -22) (max range)
	movlw	83
	subwf	ADDN_S1,w
	movlw	83              ; maximum
	btfsc	STATUS,C
	movwf	ADDN_S1			; set at 83 if >83
	
; set S_CURVE value accordingly	
	movf	ADDN_S1,w		; get value
    addlw   34h             ; address for PWM table plus W
    movwf   FSR1L           ; EEPROM address
    movlw   70h
    movwf   FSR1H
    moviw   0[FSR1]
    movwf	S_CURVE
  	goto	_RUN

PRESSURE:; pressure correction
  
; check if no sensor. AND5 is held at 0
; OUT+ (PRESSH0,1) (at AND5) is at 0V. Normally over 0.25V
; compare if >0V
    bcf     STATUS,GIE      ; interrupt off
	movf	PRESSH0,w 		; ms 
	sublw	0
	movwf	TEMP
	movf	PRESSH1,w		; ls  
	sublw	10              ; ls  ~50mV
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then pressh>0 
	return					; no sensor	

; subtract offset from pressure sensor value
	movf	PRESSL0,w		; ms lower 
   	subwf	PRESSH0,w		; ms upper  
	movwf	AARGB0
	movf	PRESSL1,w		; ls lower 
   	subwf	PRESSH1,w		; ls upper  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry
	btfsc	AARGB0,7		; if set then use 100% of value
	return      ; no compenstion below atmospheric pressure
                ; as pressure should alway be higher in exhaust

; Positive pressure above atmospheric
; divide by 16
                
    bsf     STATUS,GIE      ; interrupt on            
    bcf     STATUS,C
    rrf     AARGB0,f
    rrf     AARGB1,f       ; /2
    bcf     STATUS,C
    rrf     AARGB0,f
    rrf     AARGB1,f       ; /4
    bcf     STATUS,C
    rrf     AARGB0,f
    rrf     AARGB1,f       ; /8
    bcf     STATUS,C
    rrf     AARGB0,f
    rrf     AARGB1,f       ; /16

	movf	AARGB1,w
	sublw	28
	movlw	28              ; set at 28 max	
	btfss	STATUS,C
	movwf	AARGB1			; if >28 set at 28
	movf	AARGB1,w	

    btfsc	RICHLEAN,0		; check rich or lean
	goto	PRESSURE_RICH_SET
; lean
    addlw   89h             ; address for PWM table plus W
    movwf   FSR1L           ; EEPROM address 
    goto	LOOK            ; lookup table
PRESSURE_RICH_SET:
    addlw   0A6h            ; address for PWM table plus W
    movwf   FSR1L           ; EEPROM address
LOOK:
    movlw   70h
    movwf   FSR1H
    moviw   0[FSR1]   
    
; mult (Ip: TEMPB0, TEMPB1 by w plus divide)
MULT_CORR:
; multiply
	movwf	AARGB1
 	clrf	AARGB0
	movf	TEMPB0,w
	movwf	BARGB0
	movf	TEMPB1,w		; multiply value
	movwf	BARGB1
	
	call	FXM1616U		; multiply

; divide
	movlw   100
	movwf	BARGB1
	clrf	BARGB0
	call	FXD3216U		; divide 

; result to TEMPB0,1	
	movf	AARGB2,w 		; ms 
	movwf	TEMPB0
	movf	AARGB3,w
	movwf	TEMPB1
	return					; correction completed
	
; *****************************************************************************************
; Subroutines
    
; subroutine to wait for A/D conversion
ACQUIRE_AD1: ; interrupt version
    call    ACQUIRE_AD0
    movf	ADRESL,w
	movwf	TEMPD0		; ls
	movf	ADRESH,w
	BANKSEL PORTA		; bank0
	return
ACQUIRE_AD: ; main code 
    call    ACQUIRE_AD0
    movf	ADRESL,w
	movwf	TEMPD1		; ls
	movf	ADRESH,w
	BANKSEL PORTA		; bank0
	return    

ACQUIRE_AD0:; common code for A/D
    BANKSEL ADPCH
 	movwf	ADPCH		; set channel

; wait >8us to charge input capacitance/ 
	movlw	40
	movwf	STORE3
WAIT2C1:
	decfsz	STORE3,f
	goto	WAIT2C1	
	bsf		ADCON0,0	; ADGO bit start conversion
WAIT_CONV:
	btfsc	ADCON0,0	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return
 
; battery AND4
AND4_1:

    bcf     INTCON,GIE
	movlw	0011100B	; AND4
	call	ACQUIRE_AD
	movwf	BATT0		; ms
	movf	TEMPD1,w
	movwf	BATT1		; ls
    bsf     INTCON,GIE
	return
	
; Math routines
    
; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN_0,BIN_1. Most significant byte is BIN_0  
; Result in BCD is in BCD_0 BCD_1 & BCD_2.  
; BCD_0 is MSB, BCD_2 is LSB

BCD:
    bcf     STATUS,C	; clear carry bit
	movlw	16
    BANKSEL BCD_0       ; bank 1
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0
    clrf    BCD_2
LOOPBCD:
    rlf     BIN_1,f		; shift left binary registers
    rlf     BIN_0,f
    rlf     BCD_2,f
	rlf     BCD_1,f		; 
	rlf     BCD_0,f		; shift left BCD registers
	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
    BANKSEL PORTA 
	return              ; completed decimal to BCD operation

; decimal adjustment

DECADJ:
; for	BCD_2           ; BCD LSB address
    movlw	3           ; w has 03 
	addwf	BCD_2,w		; add 03 to BCD2 register 
	movwf	TEMPN		; store w
	btfsc	TEMPN,3		; test if >7
	movwf	BCD_2		; save in LS nibble
	movlw	30h         ; 30 for MS nibble
	addwf	BCD_2,w		; add 30 to BCD2 register
	movwf	TEMPN		; store w in temp
	btfsc	TEMPN,7		; test if >7
	movwf	BCD_2		; save in MS nibble
; for	BCD_1           ; BCD mid address
    movlw	3           ; w has 03 
	addwf	BCD_1,w		; add 03 to BCD1 register 
	movwf	TEMPN		; store w
	btfsc	TEMPN,3		; test if >7
	movwf	BCD_1		; save in LS nibble
	movlw	30h         ; 30 for MS nibble
	addwf	BCD_1,w		; add 30 to BCD1 register
	movwf	TEMPN		; store w in temp
	btfsc	TEMPN,7		; test if >7
	movwf	BCD_1		; save in MS nibble
; for	BCD_0           ; BCD MSB address
; the following is not required for this application as we don't overrange into BCD_0    
;   movlw	3           ; w has 03 
;	addwf	BCD_0,w		; add 03 to BCD0 register 
;	movwf	TEMPN		; store w
;	btfsc	TEMPN,3		; test if >7
;	movwf	BCD_0		; save in LS nibble
;	movlw	30h          ; 30 for MS nibble
;	addwf	BCD_0,w		; add 30 to BCD0 register
;	movwf	TEMPN		; store w in temp
;	btfsc	TEMPN,7		; test if >7
;	movwf	BCD_0		; save in MS nibble
  
	goto	LOOPBCD

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;       Output: product AARGxBARG in AARG
;       16x16 Bit Unsigned Fixed Point Multiply 
;       Input:  16 bit unsigned fixed point multiplicand in AARGB0,1
;               16 bit unsigned fixed point multiplier in BARGB0,1
;       Use:    CALL    FXM1616U
;       Output: 32 bit unsigned fixed point product in AARGB0,1,2,3

FXM1616U: 
			    clrf    AARGB2          ; clear partial product
                clrf    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMP160
                MOVF    AARGB1,W
                MOVWF   TEMP161
                MOVLW   08
                MOVWF   LOOPCOUNT
LOOPUM1616A:    RRF     BARGB1, F
                BTFSC   STATUS,C
                bra    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                bra    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B:    RRF     BARGB0, F
                BTFSC   STATUS,C
                bra    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                bra    LOOPUM1616B
                clrf    AARGB0
                clrf    AARGB1
				RETLW   00
BLUM1616NAP:    BCF     STATUS,C
                bra    BLUM1616NA
ALUM1616NAP:    BCF     STATUS,C
                bra    ALUM1616NA
ALOOPUM1616:    RRF     BARGB1, F
                BTFSS   STATUS,C
                bra    ALUM1616NA
                MOVF    TEMP161,W
                ADDWF   AARGB1, F
                MOVF    TEMP160,W
                BTFSC   STATUS,C
                INCFSZ  TEMP160,W
                ADDWF   AARGB0, F
ALUM1616NA:     RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                bra   ALOOPUM1616
                MOVLW  08
                MOVWF  LOOPCOUNT
BLOOPUM1616:    RRF    BARGB0, F
                BTFSS  STATUS,C
                bra   BLUM1616NA
                MOVF   TEMP161,W
                ADDWF  AARGB1, F
                MOVF   TEMP160,W
                BTFSC  STATUS,C
                INCFSZ TEMP160,W
                ADDWF  AARGB0, F
BLUM1616NA:     RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                bra   BLOOPUM1616
				return
                
; 32/16 Bit Unsigned Fixed Point Divide 32/16 -> 32.16
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 16 bit unsigned fixed point divisor in BARGB0, BARGB1
; Use: CALL FXD3216U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 16 bit unsigned fixed point remainder in REMB0, REMB1
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 2+699+2 = 703 clks
; Min Timing: 2+663+2 = 667 clks

FXD3216U: 
	clrf REMB0
	clrf REMB1
	clrf TEMP
	RLF AARGB0,W
	RLF REMB1, F
	MOVF BARGB1,W
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	RLF AARGB0, F
	MOVLW 7
	MOVWF LOOPCOUNT
LOOPU3216A: 
	RLF AARGB0,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB0,0
	BRA UADD26LA
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26LA
UADD26LA: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26LA:
	RLF AARGB0, F
	DECFSZ LOOPCOUNT, F
	BRA LOOPU3216A
	RLF AARGB1,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB0,0
	BRA UADD26L8
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26L8
UADD26L8: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26L8:
	RLF AARGB1, F
	MOVLW 7
	MOVWF LOOPCOUNT
LOOPU3216B:
	RLF AARGB1,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB1,0
	BRA UADD26LB
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26LB
UADD26LB:
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26LB: 
	RLF AARGB1, F
	DECFSZ LOOPCOUNT, F
	BRA LOOPU3216B
	RLF AARGB2,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB1,0
	BRA UADD26L16
    SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26L16
UADD26L16: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26L16: 
	RLF AARGB2, F
	MOVLW 7
	MOVWF LOOPCOUNT
LOOPU3216C: 
	RLF AARGB2,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB2,0
	BRA UADD26LC
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26LC
UADD26LC: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26LC: 
	RLF AARGB2, F
	DECFSZ LOOPCOUNT, F
	BRA LOOPU3216C
	RLF AARGB3,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB2,0
	BRA UADD26L24
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26L24
UADD26L24: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F

UOK26L24: 	
	RLF AARGB3, F
	MOVLW 7
	MOVWF LOOPCOUNT
LOOPU3216D: 
	RLF AARGB3,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB3,0
	BRA UADD26LD
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW 1
	SUBWF TEMP, F
	BRA UOK26LD
UADD26LD: 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW 1
	ADDWF TEMP, F
UOK26LD: 
	RLF AARGB3, F
	DECFSZ LOOPCOUNT, F
	BRA LOOPU3216D
	BTFSC AARGB3,0
	BRA UOK26L
	MOVF BARGB1,W
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
UOK26L:
	return

                
;Wideband data	
	
;Lambda	O2 concentration %	Ip mA	A/D value of Span relative to Vs/Ip @ ANA4 and Ip sense @ ANA6	0-5V curve output value(PWM decimal value)
				
;Standard range 0.7 to 1.84				
; lambda O2%	Ip		ANA4-ANA6			TP16			
;1.84	9.24%	1.07 mA	1.685V (D345)	5V (D1023)
;1.80	9%	1.036 mA	1.633V (D334)	4.83V (D988)
;1.73	8.5%	0.975 mA	1.536V (D314)	4.52V (D925)
;*1.7	8.29%	0.95mA mA	1.50V (D306)	4.39V (D898)
;1.658	8%	0.9158 mA	1.443V (D295)	4.21V (D861)
;1.594	7.5%	0.8568 mA	1.35V (D276)	3.92V (D803)
;1.5346	7%	0.7979 mA	1.257V (D257)	3.67V (D750)
;1.48	6.5%	0.73895 mA	1.164V (D238)	3.43V (D701)
;*1.43	6.0%	0.68 mA	1.07V (D219)	3.21V (D656)
;1.38	5.5%	0.6233 mA	0.982V (D201)	2.99V (D611)
;1.33	5%	0.5666 mA	0.893V (D183)	2.77V (D566)
;1.2916	4.5%	0.510 mA	0.804V (D164)	2.60V (D531)
;1.216	4%	0.45338 mA	0.714V (D146)	2.42V (D495)
;1.214	3.5%	0.39666 mA	0.625V (D128)	2.26V (D462)
;*1.18	3%	0.34 mA	0.54V (D109)	2.11V (D431)
;1.1445	2.5%	0.28333 mA	0.446V (D91)	1.95V (D399)
;1.112	2%	0.22666 mA	0.3571V (D73)	1.81V (D370)
;1.08227	1.5%	0.170 mA	0.2678V (D55)	1.68V (D343)
;1.0534	1%	0.1133 mA	0.1786V (D37)	1.55V (D317)
;1.026	0.5%	0.0566 mA	0.0893V (D18)	1.43V (D292)

;*1.016	0%	0 mA	0V (D0)	1.35V (D277)
				
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)
;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)
;*0.7		-1.82 mA	2.87V (D587)	0V (D0)			
				


;*denotes tabled values Intermediate values use interpolation using the O2 concentration 
;against Ip graph and the lambda calculation ((XO2/3) +1)/1-4.76XO2.

;Span is Ip x 61.9 ohm x gain (560k/22k)



	end _START
